<?php

namespace UtillzCore\Inc\Src\Listing;

use \UtillzCore\Inc\Src\Listing_Type\Listing_Type;
use \UtillzCore\Inc\Src\Woocommerce\Packages\Plan;
use \UtillzCore\Inc\Src\Woocommerce\Packages\Promotion;

class Listing {

    public $id;
    public $post;
    public $type;
    public $reviews;

    public $class = [];
    public $attrs = [];

    function __construct( $id = null ) {

        if( get_post_type( $id ) !== 'ulz_listing' ) {
            return;
        }

        $this->id = $id ? $id : get_the_ID();
        $this->post = get_post( $this->id );
        $this->type = new Listing_Type( $this->get('ulz_listing_type') );
        $this->reviews = new Reviews( $this->id );

    }

    public function get_status() {
        return get_post_status( $this->id );
    }

    public function get_author() {
        $author_id = get_post_field( 'post_author', $this->id );

        $author = [
            'id' => $author_id,
            'name' => get_the_author_meta( 'display_name', $author_id ),
            'url' => get_author_posts_url( $author_id ),
        ];

        return (object) $author;
    }

    public function get( $key ) {

        if( $key == 'post_title' ) {
            return get_the_title( $this->id );
        }

        if( $key == 'post_content' ) {
            return get_the_content( null, false, $this->id );
        }

        return Ucore()->get( $key, $this->id );

    }

    public function get_gallery( $size = 'ulz_listing', $field = 'ulz_gallery' ) {

        $gallery_attrs = Ucore()->jsoning( $field, $this->id );
        $gallery = [];

        foreach( $gallery_attrs as $image_attrs ) {
            if( isset( $image_attrs->id ) ) {
                if( is_array( $size ) ) {
                    $gallery_items = [];
                    foreach( $size as $size_item ) {
                        if( $image = Ucore()->get_image( $image_attrs->id, $size_item ) ) {
                            $gallery_items[ $size_item ] = $image;
                        }
                    }
                    $gallery[] = $gallery_items;
                }else{
                    if( $image = Ucore()->get_image( $image_attrs->id, $size ) ) {
                        $gallery[] = $image;
                    }
                }
            }
        }

        return $gallery;

	}

    public function get_first_from_gallery( $size = 'ulz_listing' ) {

        $gallery = $this->get_gallery( $size );

        if( isset( $gallery[0] ) ) {
            return $gallery[0];
        }

        return null;

	}

    public function get_details( $key ) {

        $details = [];

        foreach( Ucore()->jsoning( $key, $this->type->id ) as $detail ) {

            // taxonomy
            if( $detail->template->id == 'taxonomy' ) {

                $value = Ucore()->get( Ucore()->prefix( $detail->fields->key ), $this->id, false );

                if( is_array( $value ) ) {
                    foreach( $value as $val ) {

                        $term = get_term( $val, $detail->fields->key );

                        if( is_object( $term ) && ! is_wp_error( $term ) ) {

                            $term_icon = get_term_meta( $term->term_id, 'ulz_icon', true );

                            $details[] = (object) [
                                'content' => $term->name,
                                'icon' => ( $detail->fields->display_icons && $term_icon ) ? $term_icon : null
                            ];

                        }
                    }
                }
            }

            // label
            elseif( $detail->template->id == 'label' ) {

                if( $value = $this->get( Ucore()->prefix( $detail->fields->key ) ) ) {

                    if( isset( $detail->fields->render_format ) ) {
                        switch( $detail->fields->render_format ) {
                            case 'price':
                                $value = Ucore()->format_price( $value );
                                break;
                        }
                    }

                    $content = $detail->fields->format ? str_replace( '{field}', $value, $detail->fields->format ) : $value;

                    $details[] = (object) [
                        'content' => $content,
                        'icon' => isset( $detail->fields->icon ) ? $detail->fields->icon : null
                    ];
                }
            }

        }

        return $details;

    }

    public function get_field_item( $key ) {

        foreach( Ucore()->json_decode( $this->type->get('ulz_fields') ) as $field ) {
            if( isset( $field->fields->key ) && Ucore()->prefix( $field->fields->key ) == $key ) {
                return $field;
            }
        }

        return null;

    }

    public function awaits_payment() {

        wp_update_post([
            'ID' => $this->id,
            'post_status' => 'pending_payment'
        ]);

    }

    public function pack_away( $user_plan_id ) {

        /*
         * attach listing to plan
         * and +1 to plan counter
         *
         */
        add_post_meta( $user_plan_id, 'ulz_attached', $this->id );
        update_post_meta( $user_plan_id, 'ulz_count', (int) get_post_meta( $user_plan_id, 'ulz_count', true ) + 1 );

        /*
         * attach plan to listing
         *
         */
        delete_post_meta( $this->id, 'ulz_plan' );
        add_post_meta( $this->id, 'ulz_plan', $user_plan_id );

        // set expiration
        $this->set_expiration();

        // set priority
        $priority = (int) get_post_meta( $user_plan_id, 'ulz_priority', true );
        if( $priority > 0 ) {
            update_post_meta( $this->id, 'ulz_priority', $priority );
        }

        // change listing status
        wp_update_post([
            'ID' => $this->id,
            'post_status' => Ucore()->get( 'ulz_requires_admin_approval', $this->type->id ) ? 'pending' : 'publish'
        ]);

    }

    static public function calculate_expiration( $duration ) {

        if ( $duration ) {
    		return date( 'Y-m-d', strtotime( "+{$duration} days", current_time( 'timestamp' ) ) );
    	}

    	return '';

    }

    public function set_expiration() {

        $listing_plan_id = $this->get( 'ulz_plan' );

        if( $listing_plan_id ) {

            $plan_id = Ucore()->get( 'ulz_product_id', $listing_plan_id );
            $plan = new Plan( $plan_id );

            if( $plan->id ) {

                $duration = $plan->get_duration();
                if( $duration > 0 ) {

                    delete_post_meta( $this->id, 'ulz_listing_expires' );
                    add_post_meta( $this->id, 'ulz_listing_expires', Listing::calculate_expiration( $duration ) );

                }
            }
        }
    }

    public function get_author_id() {
        return $this->post->post_author;
    }

    public function is_owner() {

        if( get_current_user_id() == $this->get_author_id() ) {
            return true;
        }

        return;

    }

    public function promote( $user_promotion_id ) {

        $promotion = new Promotion( Ucore()->get( 'ulz_product_id', $user_promotion_id ) );
        if( ! $promotion->id ) {
            return;
        }

        $expires = Ucore()->get( 'ulz_expires', $user_promotion_id );
        if( $expires <= time() ) {
            return;
        }

        $duration = $promotion->get_duration();
		if( $duration == 0 ) {
			$duration = 30;
		}

		$priority = $promotion->get_priority();
		if( $priority == 0 ) {
			$priority = 2;
		}

        // attach listing to promotion
        update_post_meta( $user_promotion_id, 'ulz_attached', $this->id );

        // expiration
        update_post_meta( $this->id, 'ulz_promotion_expires', $expires );

        // priority
        $old_priority = (int) get_post_meta( $this->id, 'ulz_priority', true );
        update_post_meta( $this->id, 'ulz_old_priority', $old_priority );
        update_post_meta( $this->id, 'ulz_priority', $priority );
        delete_post_meta( $this->id, 'ulz_priority_custom' );

        switch( $priority ) {
            case 0: update_post_meta( $this->id, 'ulz_priority_selection', 'normal' ); break;
            case 1: update_post_meta( $this->id, 'ulz_priority_selection', 'featured' ); break;
            case 2: update_post_meta( $this->id, 'ulz_priority_selection', 'promoted' ); break;
            default:
                update_post_meta( $this->id, 'ulz_priority_selection', 'custom' );
                update_post_meta( $this->id, 'ulz_priority_custom', $priority );
        }

    }

    public function the_classes() {

        global $ulz_listing;

        $class = $this->class;
        $priority = (int) $this->get('ulz_priority');

        switch( $priority ) {
            case 0: $class[] = 'ulz--normal'; break;
            case 1: $class[] = 'ulz--featured'; break;
            case 2: $class[] = 'ulz--promoted'; break;
        }

        $cover_type = $ulz_listing->type->get('ulz_listing_cover_type');
        $class[] = sprintf( 'ulz-cover--%s', $cover_type );

        echo ' ' . implode( ' ', apply_filters( 'utillz/explore/listing/classes', $class ) );

    }

    public function the_attrs() {

        $attrs = $this->attrs;
        $attrs['data-id'] = $this->id;
        $attrs['data-priority'] = (int) $this->get('ulz_priority');

        echo Ucore()->attrs( apply_filters( 'utillz/explore/listing/attributes', $attrs ) );

    }

    public function report( $reason ) {

        $report_id = wp_insert_post([
			'post_title' => '',
			'post_status' => 'pending',
			'post_type' => 'ulz_report',
			'post_author' => get_current_user_id(),
            'meta_input'  => [
				'ulz_listing_id' => $this->id,
				'ulz_report_reason' => $reason,
            ]
		]);

        wp_update_post([
            'ID' => $report_id,
            'post_title' => sprintf( '#%s, %s', $report_id, get_the_title( $this->id ) ),
        ]);

        return $report_id;

    }

    public function has_reported() {

        $query = new \WP_Query([
            'post_status' => 'any',
            'post_type' => 'ulz_report',
            'post_author' => get_current_user_id(),
            'meta_query' => [
                [
                    'key' => 'ulz_listing_id',
                    'value' => $this->id,
                    'compare' => '=',
                ]
            ]
        ]);

        return $query->found_posts > 0;

    }

    public function get_priority() {
        return $this->get('ulz_priority');
    }

    public function get_priority_label() {

        $label = $this->get_priority();
        $labels = Ucore()->get_priority_labels();

        if ( isset( $labels[ $label ] ) ) {
    		return $labels[ $label ];
    	}

    }

    public function get_pricing( $addons = [] ) {

        $output = [
            'base' => 0,
            'service_fee' => 0,
            'extras' => [],
            'extras_total' => 0,
            'addons' => [],
            'addons_total' => 0,
            'payment_processing' => 0,
            'payment_processing_name' => '',
            'total' => 0,
            'processing' => 0,
        ];

        // get base
        $base = floatval( $this->get('ulz_price') );
        $output['base'] = round( $base, 2 );

        // extra services: fee, per day
        $extras = json_decode( $this->get('ulz_extra_pricing') );
        if( $extras ) {
            foreach( $extras as $extra ) {
                $output['extras'][] = $extra->fields;
                $extra_price = floatval( $extra->fields->price );
                $output['extras_total'] += $extra_price;
            }
        }

        $output['extras_total'] = round( $output['extras_total'], 2 );

        // addons
        if( $this->type->get('ulz_allow_addons') ) {
            if( is_array( $addons ) ) {
                $available_addons = Ucore()->jsoning( 'ulz_addons', $this->id );
                if( is_array( $available_addons ) ) {
                    foreach( $available_addons as $addon_item ) {
                        if( in_array( $addon_item->fields->key, $addons ) ) {
                            $addon_price = floatval( $addon_item->fields->price );
                            $output['addons'][ $addon_item->fields->key ] = (object) [
                                'name' => $addon_item->fields->name,
                                'price' => $addon_price,
                            ];
                            $output['addons_total'] += $addon_price;
                        }
                    }
                }
            }
        }

        // service fee
        $service_fee_type = $this->type->get('ulz_service_fee_type');
        switch( $service_fee_type ) {
            // fixed
            case 'fixed':
                $output['service_fee'] = round( $this->type->get('ulz_service_fee_amount_fixed'), 2 );
                break;
            // percentage
            case 'percentage':
                $service_fee_amount_percentage = floatval( $this->type->get('ulz_service_fee_amount_percentage') );
                if( $service_fee_amount_percentage > 0 ) {
                    $output['service_fee'] = round( ( $service_fee_amount_percentage / 100 ) * $output['base'], 2 );
                }
                break;
        }

        // payment processing info
        $payment_processing = $this->type->get('ulz_payment_processing');
        $payment_processing_types = Ucore()->get_payment_processing_types();
        $output['payment_processing'] = $payment_processing;
        $output['payment_processing_name'] = $payment_processing_types[ $payment_processing ];

        // total
        $output['total'] =
            $output['base'] +
            $output['service_fee'] +
            $output['extras_total'] +
            $output['addons_total'];

        // processing
        switch( $payment_processing ) {
            case 'full':
                $output['processing'] = $output['total'];
                break;
            case 'percentage':
                $payment_processing_percentage = floatval( $this->type->get('ulz_payment_processing_percentage') );
                if( $payment_processing_percentage > 0 ) {
                    $output['processing'] = round( ( $payment_processing_percentage / 100 ) * $output['total'], 2 );
                }else{
                    $output['processing'] = $output['total'];
                }
                break;
            case 'service_fee':
                $output['processing'] = $output['service_fee'];
                break;
            case 'locally':
                break;
        }

        return (object) apply_filters('utillz/listing/get-pricing', $output, $this, $addons);

    }

    public function get_address() {
		return $this->get('ulz_location');
	}

    public function get_lat() {
		return $this->get('ulz_location__lat');
	}

	public function get_lng() {
        return $this->get('ulz_location__lng');
	}

    public function get_coordinates() {
		return $this->get_lat() . ',' . $this->get_lng();
	}

    /*
	 * generate json-ld data-structured
	 *
	 */
	public function get_json_ld() {

        // check for listing type
        if( ! $this->type->id ) {
            return;
        }

        if( $type_slug = $this->type->get('ulz_slug') ) {

            // use custom handler through the hook to render any schema
    		return apply_filters( "utillz/listing-type/{$type_slug}/jsonld", null, $this );

        }
	}

    /*
     * example to attach any structured data to a listing type
     *
     *
    add_filter('utillz/listing-type/[your-listing-type-slug]/jsonld', function( $structured_data, $listing ) {

        return [
            '@context' => 'https://schema.org/',
            '@type' => $listing->type->get('ulz_name'), // listing type name
            'name' => $listing->post->post_title, // listing title
            'description' => wp_trim_words( $listing->post->post_content, 55 ), // listing content
            'image' => get_the_post_thumbnail_url( $listing->id,'medium'), // featured image
        ];
    }, 10, 2 );*/

    public function get_action( $key = '' ) {

        if( ! $this->type ) {
            return null;
        }

        return $this->type->get_action( $key );

    }

    public function get_action_product( $action = 'purchase' ) {

        $action_product = $this->type->get_action_type( $action );

        if( ! $action_product ) {
            return;
        }

        if( isset( $action_product->fields->action_type_product ) ) {
            if( is_array( $action_product->fields->action_type_product ) ) {
                $products = [];
                foreach( $action_product->fields->action_type_product as $product_id ) {
                    $products[ $product_id ] = wc_get_product( $product_id );
                }
                return array_filter( $products );
            }else{
                return wc_get_product( $action_product->fields->action_type_product );
            }
        }

    }

    public function get_price() {

        $base = floatval( $this->get('ulz_price') );
        $service_fee_type = $this->type->get('ulz_service_fee_type');
        $fee = 0;

        switch( $service_fee_type ) {
            case 'fixed':
                $fee = floatval( $this->type->get('ulz_service_fee_amount_fixed') );
                break;
            case 'percentage':
                $fee = floatval( $this->type->get('ulz_service_fee_amount_percentage') ) * $base * 0.01;
                break;
        }

        return [
            'service_fee_type' => $service_fee_type,
            'base' => round( $base, 2 ),
            'fee' => round( $fee, 2 ),
            'final' => round( $base + $fee, 2 )
        ];

    }

    public function get_views() {

        global $wpdb;

        return (int) $wpdb->get_var(
            $wpdb->prepare("
                    SELECT SUM( count )
                    FROM {$wpdb->prefix}utillz_views
                    WHERE listing_id = %d
                ",
                $this->id
            )
        );

    }

}
