<?php

namespace UtillzCore\Inc\Src\Woocommerce;

use \UtillzCore\Inc\Src\Request\Request;
use \UtillzCore\Inc\Src\Listing\Listing;
use \UtillzCore\Inc\Src\Traits\Singleton;
use \UtillzCore\Inc\Src\Woocommerce\Packages\Plan;
use \UtillzCore\Inc\Src\Woocommerce\Packages\Promotion;
use \UtillzCore\Inc\Src\Wallet;

class Order {

    use Singleton;

    function __construct() {

        add_action( 'woocommerce_order_status_completed', [ $this, 'plan_order_completed' ] );
        add_action( 'woocommerce_order_status_completed', [ $this, 'promotion_order_completed' ] );
        add_action( 'woocommerce_order_status_completed', [ $this, 'purchase_order_completed' ] );

        add_filter( 'woocommerce_order_item_display_meta_key', [ $this, 'display_meta_key' ], 20, 3 );
        add_filter( 'woocommerce_order_item_display_meta_value', [ $this, 'display_meta_value' ], 20, 3 );

    }

    public function plan_order_completed( $order_id ) {

		$order = wc_get_order( $order_id );
        $user_id = $order->get_customer_id();

        if( ! $user_id ) {
            return;
        }

        // prevent duplication
        if( get_post_meta( $order_id, 'utillz_payment_plan_processed', true ) ) {
			return;
		}

		foreach( $order->get_items() as $item ) {

			$product = wc_get_product( $item['product_id'] );

            /*
             * plan
             *
             */
            if( $product->is_type( 'listing_plan' ) ) {

                $plan = new Plan( $product->get_id() );

                /*
                 * inser user plan post
                 *
                 */
                $order = wc_get_order( $order_id );
                $user_plan_id = $plan->create( $order_id, $order->get_user_id() );

    			for( $i = 0; $i < $item['qty']; $i++ ) {

                    /*
                     * handle each listing inside plan
                     *
                     */
                     $listing_ids = json_decode( $item->get_meta('_listing_id') );

                     if( is_array( $listing_ids ) ) {
                         foreach( $listing_ids as $id ) {

                             $listing = new Listing( $id );

                             if( $listing->id && $listing->get_status() == 'pending_payment' ) {
                                 $listing->pack_away( $user_plan_id );
                             }
                         }
                     }
    			}
            }
		}

		update_post_meta( $order_id, 'utillz_payment_plan_processed', true );

	}

    public function promotion_order_completed( $order_id ) {

		$order = wc_get_order( $order_id );
        $user_id = $order->get_customer_id();

        // prevent duplication
        if( get_post_meta( $order_id, 'utillz_payment_promotion_processed', true ) ) {
			return;
		}

		foreach( $order->get_items() as $item ) {

			$product = wc_get_product( $item['product_id'] );

			if( $product->is_type( 'listing_promotion' ) && $user_id ) {
				for ( $i = 0; $i < $item['qty']; $i ++ ) {

                    $listing_ids = json_decode( $item->get_meta('_listing_id') );
                    $package_id = $item->get_product()->get_id();
                    $promotion = new Promotion( $package_id );

                    if( is_array( $listing_ids ) ) {
                        foreach( $listing_ids as $id ) {

                            $listing = new Listing( $id );

                            if( $listing->id ) {

                                $user_promotion_id = $promotion->create( $order_id );
                                $listing->promote( $user_promotion_id );

                            }
                        }
                    }
				}
			}
		}

		update_post_meta( $order_id, 'utillz_payment_promotion_processed', true );

	}

    public function purchase_order_completed( $order_id ) {

        $order = wc_get_order( $order_id );
        $user_id = $order->get_customer_id();

        if( ! $user_id ) {
            return;
        }

        // prevent duplication
        if( get_post_meta( $order_id, 'utillz_payment_purchase_processed', true ) ) {
			return;
		}

		foreach( $order->get_items() as $item ) {

            $product = wc_get_product( $item['product_id'] );

            if( $product->is_type( 'listing_purchase' ) ) {

                for( $i = 0; $i < $item['qty']; $i++ ) {

                    $listing_ids = json_decode( $item->get_meta('_listing_id') );

                    if( isset( $listing_ids[0] ) ) {

                        $listing = new Listing( $listing_ids[0] );

                        if( $listing->id ) {

                            $user = get_userdata( (int) $item->get_meta('_request_user_id') );

                            /*
                             * create entry
                             *
                             */
                            $entry_id = wp_insert_post([
                                'post_title' => $user->display_name,
                                'post_status' => 'publish',
                                'post_type' => 'ulz_entry',
                                'post_author' => $listing->post->post_author,
                                'meta_input' => [
                                    'ulz_entry_type' => 'purchase',
                                    'ulz_listing' => $listing->id,
                                    'ulz_request_user_id' => (int) $item->get_meta('_request_user_id'),
                                    'ulz_pricing' => wp_slash( json_encode( $item->get_meta('_pricing') ) ),
                                    'ulz_order_id' => $order_id,
                                ]
                            ]);

                            // entry was created successfully
                            if( ! is_wp_error( $entry_id ) ) {

                                /*
                                 * add funds
                                 *
                                 */
                                $pricing = $item->get_meta('_pricing');
                                $price = $listing->get_price();

                                // add to owner wallet
                                $wallet = new Wallet( $listing->post->post_author );

                                $earnings = number_format( $pricing->total - $pricing->service_fee, 2, '.', '' );

                                // charge host fee
                                if( $listing->type->get('ulz_host_fee_type') == 'percentage' ) {
                                    if( $host_fee = floatval( $listing->type->get('ulz_host_fee_amount_percentage') ) ) {
                                        $earnings = number_format( ( ( 100 - $host_fee ) * 0.01 ) * $earnings, 2 , '.', '' );
                                    }
                                }

                                $wallet->add_funds( $earnings, $order_id, 'earnings' );

                                /*
                                 * send notification
                                 *
                                 */
                				utillz_core()->notify->distribute( 'new-purchase', [
                					'user_id' => $listing->post->post_author,
                					'meta' => [
                                        'entry_id' => $entry_id,
                                        'listing_id' => $listing->id,
                                        'from_user_id' => (int) $item->get_meta('_request_user_id'),
                                    ],
                				]);

                            }

                        }
                    }
                }
            }
        }

        update_post_meta( $order_id, 'utillz_payment_purchase_processed', true );

    }

    public function display_meta_key( $key, $meta, $item ) {

        if( $meta->key == '_listing_id' ) {
            return __( 'Listings Attached', 'utillz-core' );
        }

        return $key;

    }

    public function display_meta_value( $value, $meta, $item ) {

        if( $meta->key == '_listing_id' ) {
            $array = json_decode( $meta->value );
            if( is_array( $array ) ) {
                $arr = [];
                foreach( $array as $id ) {
                    $listing = new Listing( $id );
                    if( $listing->id ) {
                        $title = get_the_title( $listing->id );
                        $arr[] = '#' . $listing->id . ' - ' . ( $title ? $title : esc_html__( '( no title )', 'utillz-core' ) );
                    }
                }
                return implode( ', ', $arr );
            }
        }

        return $value;

    }

}
